<?php

require 'Curl.php';
require 'Helper.php';
require 'Log.php';
require 'GrsApiConfig.php';

class GrsFido2ClientService
{

	protected $grsApiConfig;
	protected $fido2Servers;
	protected $helper;
	protected $clientID;
	protected $sharedSecret;
	protected $logger;

	public function __construct($fido2Servers,$clientID,$secret,$logFile='/var/log/fido2.log')
	{
		$this->grsApiConfig = new GrsApiConfig();
		$this->fido2Servers = $fido2Servers;
		$this->clientID = $clientID;
		$this->sharedSecret = $secret;
		$this->helper = new Helper();
		try
		{
			$this->logger = new Log($logFile);
		}
		catch(Exception $e)
		{
			throw new Exception($e->getMessage());
		}
	}

	public function proxyFido2Request($payload)
	{
		if(empty($this->sharedSecret))
		{
			$err = 'Shared secret not configured on RP';
			$this->logger->log('ERROR',$err);
			return json_encode(['success' => false, 'msg' => $err]);
		}

		if(empty($this->clientID))
		{
			$err = 'Client ID not configured on RP';
			$this->logger->log('ERROR',$err);
			return json_encode(['success' => false, 'msg' => $err]);
		}
		
		if(empty($this->fido2Servers))
		{
			$err = 'FIDO2 server hostname not configured on RP';
			$this->logger->log('ERROR',$err);
			return json_encode(['success' => false, 'msg' => $err]);
		}

		if(!is_array($payload))
		{
			$payload = json_decode($payload,true);
		}

		$tmpDirectory = sys_get_temp_dir();

		if($payload['operation'] == 'register-complete' || $payload['operation'] == 'authenticate-complete')
		{
			if(!array_key_exists('transaction_id',$payload))
			{
				$this->logger->log('ERROR','transaction_id not found in request');
				return json_encode(['success' => false, 'msg' => '']);
			}

			$transactionId = $payload['transaction_id'];
		}

		$curlOptions = array(
			CURLOPT_HTTPHEADER => array('Content-Type: application/json')
		);

		try
		{
			if($payload['operation'] == 'register-init')
			{
				$res = $this->initiateRegistration($payload, $curlOptions);
			}
			if($payload['operation'] == 'register-complete')
			{
				$res = $this->registrationComplete($payload, $curlOptions, $transactionId);
			}
			if($payload['operation'] == 'authenticate-init')
			{
				$res = $this->initiateAuthentication($payload, $curlOptions);
			}
			if($payload['operation'] == 'authenticate-complete')
			{
				$res = $this->completeAuthentication($payload, $curlOptions, $transactionId);
			}
			$response = $res['response'];
			$this->logger->log('DEBUG','Received response from FIDO2 server'.print_r($res, 1));
		}
		catch(Exception $e)
		{
			$msg = $e->getMessage();
			$this->logger->log('ERROR','Error: '.$msg);

			return json_encode(['success' => false, 'msg' => $msg]);
		}

		if($payload['operation'] == 'register-init' || $payload['operation'] == 'authenticate-init')
		{
			$resArr = json_decode($response,true);
		}
		return $response;
	}

	public function initiateRegistration($payload, $curlOptions){
		$grsApiUrl = $this->grsApiConfig->getFido2RegisterInitServerUrl($this->fido2Servers);
		$this->logger->log('DEBUG','url: '.$grsApiUrl);
		$requestData = array();
		$requestData['username'] = $payload['username'];
		$requestData['password'] = $payload['password'];
		$requestData['token_label'] = $payload['token_label'];
		$requestData['client_id'] = $this->clientID;

		$sharedSecretHash = $this->helper->getSharedSecretHash($this->sharedSecret, $this->clientID);
		$hash = $this->helper->computeHashHmac($requestData, $sharedSecretHash);
		$requestData['h'] = $hash;
		$jsonPayload = json_encode($requestData);

		$requestData['password'] = " ";
		$this->logger->log('DEBUG','payload: '.print_r($requestData, 1));

		$res = $this->query($grsApiUrl,$jsonPayload,$curlOptions);
		return $res;
	}

	public function registrationComplete($payload, $curlOptions, $transactionId){
		$grsApiUrl = $this->grsApiConfig->getFido2RegisterCompleteServerUrl($this->fido2Servers);
		$this->logger->log('DEBUG','url: '.$grsApiUrl);
		$requestData = array();
		$requestData['transaction_id'] = $transactionId;
		$requestData['username'] = $payload['username'];
		$requestData['client_id'] = $this->clientID;
		$requestData['authenticatorAttestationResponse'] = $payload['authenticatorAttestationResponse'];
		$sharedSecretHash = $this->helper->getSharedSecretHash($this->sharedSecret, $this->clientID);
		$hash = $this->helper->computeHashHmac($requestData, $sharedSecretHash);
		$requestData['h'] = $hash;
		$jsonPayload = json_encode($requestData);

		$this->logger->log('DEBUG','payload: '.print_r($requestData, 1));
		$res = $this->query($grsApiUrl,$jsonPayload,$curlOptions);
		return $res;
	}

	public function initiateAuthentication($payload, $curlOptions){
		$grsApiUrl = $this->grsApiConfig->getFido2AuthenticateInitServerUrl($this->fido2Servers);
		$this->logger->log('DEBUG','url: '.$grsApiUrl);
		$requestData = array();
		$requestData['username'] = $payload['username'];
		$requestData['password'] = $payload['password'];
		$requestData['client_id'] = $this->clientID;
		$sharedSecretHash = $this->helper->getSharedSecretHash($this->sharedSecret, $this->clientID);
		$hash = $this->helper->computeHashHmac($requestData, $sharedSecretHash);
		$requestData['h'] = $hash;
		$jsonPayload = json_encode($requestData);

		$requestData['password'] = " ";
		$this->logger->log('DEBUG','payload: '.print_r($requestData, 1));

		$res = $this->query($grsApiUrl,$jsonPayload,$curlOptions);
		return $res;
	}

	public function completeAuthentication($payload, $curlOptions, $transactionId){
		$grsApiUrl = $this->grsApiConfig->getFido2AuthenticateCompleteServerUrl($this->fido2Servers);
		$this->logger->log('DEBUG','url: '.$grsApiUrl);
		$requestData = array();
		$requestData['transaction_id'] = $transactionId;
		$requestData['username'] = $payload['username'];
		$requestData['client_id'] = $this->clientID;
		$requestData['authenticatorAttestationResponse'] = $payload['authenticatorAttestationResponse'];
		$sharedSecretHash = $this->helper->getSharedSecretHash($this->sharedSecret, $this->clientID);
		$hash = $this->helper->computeHashHmac($requestData, $sharedSecretHash);
		$requestData['h'] = $hash;
		
		$jsonPayload = json_encode($requestData);
		$this->logger->log('DEBUG','payload: '.print_r($requestData, 1));
		$res = $this->query($grsApiUrl,$jsonPayload,$curlOptions);
		return $res;
	}

	private function query($url,$params, $curl_options = NULL)
	{
		$curl = new Curl();
		$response = $curl->init($url)->post($params,$curl_options);
		return $response;
    }
}

?>

